package com.blogspot.m3g4h4rd.wicket;

import org.apache.wicket.Component;
import org.apache.wicket.WicketRuntimeException;
import org.apache.wicket.markup.html.form.Radio;
import org.apache.wicket.markup.html.form.RadioGroup;
import org.apache.wicket.model.IModel;
import org.apache.wicket.util.convert.ConversionException;

/**
 * This radio group class implements a special behavior for radio buttons
 * having a model value of the type boolean. When selecting a radio button
 * its model value is set to true in the case formerly set model value is 
 * of the type boolean.
 * The model value of the non-selected radio buttons is set to false, in the 
 * case the formerly set model value is of the type boolean. 
 * <br/><br/>
 * 
 * This class uses code parts of  
 * {@link org.apache.wicket.markup.html.form.RadioGroup}.
 * 
 * @author Silvio Meier
 *
 * @param <T> The type of the model value.
 */
public class BooleanRadioGroup<T> extends ExtendedRadioGroup {

	/**
	 * The constructor.
	 * @param id The markup id of the boolean radio group.
	 * @param model The model that is initially set to the boolean radio group.
	 */
	public BooleanRadioGroup(String id, IModel model) {
		super(id, model);
	}
	
	/**
	 * The constructor.
	 * @param id The markup id of the boolean radio group.
	 */
	public BooleanRadioGroup(String id) {
		super(id);
	}
	
	/**
	 * Sets the model object value of the currently selected radio button to true and
	 * set the model object values of all other radio buttons to false but only if their
	 * current value is of the type Boolean. The method returns true as the value of the 
	 * selected radio button.
	 * 
	 * @see org.apache.wicket.markup.html.form.FormComponent#convertValue(String[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	protected T convertValue(String[] input) throws ConversionException
	{
		if (input != null && input.length > 0)
		{
			final String value = input[0];

			// retrieve the selected single radio choice component
			choice = (Radio<T>)visitChildren(Radio.class,
				new Component.IVisitor<Radio<T>>()
				{
					public Object component(Radio<T> radio)
					{
						if (radio.getValue().equals(value))
						{
							return radio;
						}
						return CONTINUE_TRAVERSAL;
					}

				});

			// if we have boolean choices in the group, 
			// set the model of the non selected radios to false.
			visitChildren(Radio.class,
					new Component.IVisitor<Radio<T>>()
					{
						public Object component(Radio<T> radio)
						{
							if (radio.getDefaultModelObject() instanceof Boolean) {
								if (radio.equals(choice))
								{
									radio.setDefaultModelObject(true);									
								} else {
									radio.setDefaultModelObject(false);
								}
							}
							return CONTINUE_TRAVERSAL;
						}
					});

			
			if (choice == null)
			{
				throw new WicketRuntimeException(
					"submitted http post value [" +
						value +
						"] for RadioGroup component [" +
						getPath() +
						"] is illegal because it does not contain relative path to a Radio componnet. " +
						"Due to this the RadioGroup component cannot resolve the selected Radio component pointed to by the illegal value. A possible reason is that componment hierarchy changed between rendering and form submission.");
			}


			// assign the value of the group's model
			return (T)choice.getModelObject();
		}
		return null;
	}

}
