package com.blogspot.m3g4h4rd.wicket;



import org.apache.wicket.Component;
import org.apache.wicket.WicketRuntimeException;
import org.apache.wicket.markup.html.form.Radio;
import org.apache.wicket.markup.html.form.RadioGroup;
import org.apache.wicket.model.IModel;
import org.apache.wicket.util.convert.ConversionException;

/**
 * This radio group implementation allows to retrieve the 
 * selected radio button in the group of radio buttons. 
 * This is sometimes necessary in order to implement more 
 * complex behavior in GUIs. In contrast, the super class 
 * {@link org.apache.wicket.markup.html.form.RadioGroup} does
 * not provide a way to retrieve the selected radio button
 * but only the model of the selected radio button.
 * <br/><br/>
 * This class uses code parts of  
 * {@link org.apache.wicket.markup.html.form.RadioGroup}.
 * 
 * @author Silvio Meier
 * @param <T> The model class representing the model of the radio group.
 */
public class ExtendedRadioGroup<T> extends RadioGroup {

	/**
	 * The radio button that has been selected. 
	 */
	protected Radio<T> choice = null;
	
	/**
	 * The constructor which initializes the radio group.
	 * @param id The id of the radio group.
	 * @param model The model used initially for the 
	 * group hierarchy.
	 */
	public ExtendedRadioGroup(String id, IModel model) {
		super(id, model);
	}
	
	/**
	 * The constructor which initializes the radio group.
	 * @param id The id of the radio group.
	 */
	public ExtendedRadioGroup(String id) {
		super(id);
	}
	
	/**
	 * Apart from the selected model value, this method retrieves the 
	 * selected radio button and stores it in a variable for later 
	 * retrieval.
	 * 
	 * @see org.apache.wicket.markup.html.form.FormComponent#convertValue(String[])
	 */
	@SuppressWarnings("unchecked")
	@Override
	protected T convertValue(String[] input) throws ConversionException
	{
		if (input != null && input.length > 0)
		{
			final String value = input[0];

			// retrieve the selected single radio choice component
			choice = (Radio<T>)visitChildren(Radio.class,
				new Component.IVisitor<Radio<T>>()
				{
					public Object component(Radio<T> radio)
					{
						if (radio.getValue().equals(value))
						{
							return radio;
						}
						return CONTINUE_TRAVERSAL;
					}

				});

			if (choice == null)
			{
				throw new WicketRuntimeException(
					"submitted http post value [" +
						value +
						"] for RadioGroup component [" +
						getPath() +
						"] is illegal because it does not contain relative path to a Radio componnet. " +
						"Due to this the RadioGroup component cannot resolve the selected Radio component pointed to by the illegal value. A possible reason is that componment hierarchy changed between rendering and form submission.");
			}

			// assign the value of the group's model
			return choice.getModelObject();
		}
		return null;
	}

	/**
	 * @return Returns the selected radio button in the radio button group.
	 */
	public Radio getSelectedRadio() {
		return choice;
	}	
}